// L0 Screening Questions Generator for Recruiters

async function generateScreeningQuestions(jobDescription) {
    const prompt = `
    You are an expert technical recruiter. Generate 10 L0 (Level 0) screening questions for initial candidate filtering based on this job description.
    
    Job Description:
    ${jobDescription}
    
    Create questions that:
    1. Can be answered in 1-2 sentences
    2. Filter out unqualified candidates quickly
    3. Test must-have skills/experience
    4. Include both technical and behavioral aspects
    5. Are specific to this role (not generic)
    
    Format as JSON:
    {
        "technical": [5 technical screening questions],
        "experience": [3 experience-related questions],
        "behavioral": [2 behavioral/culture fit questions],
        "expectedAnswers": {
            "question1": "what to look for in answer",
            ...
        }
    }
    `;

    // Call your backend API
    const response = await fetch(`${BACKEND_URL}/api/generate-screening`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${authToken}`
        },
        body: JSON.stringify({ jobDescription, prompt })
    });

    return await response.json();
}

// Generate interview scorecard
async function generateScorecard(jobDescription) {
    const prompt = `
    Create an interview evaluation scorecard for this role:
    
    ${jobDescription}
    
    Include:
    1. Key competencies to evaluate (technical skills)
    2. Soft skills to assess
    3. Red flags to watch for
    4. Scoring rubric (1-5 scale)
    5. Must-have vs nice-to-have skills
    
    Format as structured JSON.
    `;

    const response = await fetch(`${BACKEND_URL}/api/generate-scorecard`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${authToken}`
        },
        body: JSON.stringify({ jobDescription, prompt })
    });

    return await response.json();
}

// Auto-generate rejection/acceptance email templates
async function generateEmailTemplates(jobDescription, candidateName, decision) {
    const prompt = `
    Generate a professional ${decision} email for candidate ${candidateName} for this role:
    
    ${jobDescription}
    
    Make it:
    1. Personalized but professional
    2. Clear about next steps
    3. Encouraging (even if rejection)
    4. Brief but complete
    `;

    const response = await fetch(`${BACKEND_URL}/api/generate-email`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${authToken}`
        },
        body: JSON.stringify({ jobDescription, candidateName, decision, prompt })
    });

    return await response.json();
}

// Bulk resume ranking
async function rankCandidates(jobDescription, resumes) {
    const prompt = `
    Rank these ${resumes.length} candidates for this role:
    
    Job Description: ${jobDescription}
    
    For each resume:
    1. Score 1-100
    2. List top 3 strengths
    3. List top 3 gaps
    4. Provide hiring recommendation (Strong Yes/Yes/Maybe/No)
    5. Suggested interview focus areas
    
    Return ranked list from best to worst match.
    `;

    const response = await fetch(`${BACKEND_URL}/api/rank-candidates`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${authToken}`
        },
        body: JSON.stringify({ jobDescription, resumes, prompt })
    });

    return await response.json();
}

// Export functions
if (typeof module !== 'undefined' && module.exports) {
    module.exports = {
        generateScreeningQuestions,
        generateScorecard,
        generateEmailTemplates,
        rankCandidates
    };
}