// Content script for extracting resume text from various sources

const ResumeExtractor = {
    // Extract text from LinkedIn profile
    extractLinkedIn() {
        const sections = {
            name: document.querySelector('.pv-text-details__left-panel h1')?.innerText || '',
            headline: document.querySelector('.pv-text-details__left-panel .text-body-medium')?.innerText || '',
            about: document.querySelector('.pv-about__summary-text')?.innerText || '',
            experience: [],
            education: [],
            skills: []
        };

        // Extract experience
        const experienceSection = document.querySelector('#experience');
        if (experienceSection) {
            const experienceItems = experienceSection.querySelectorAll('.pvs-entity');
            experienceItems.forEach(item => {
                const role = item.querySelector('.mr1 span[aria-hidden="true"]')?.innerText || '';
                const company = item.querySelector('.t-14 span[aria-hidden="true"]')?.innerText || '';
                const duration = item.querySelector('.pvs-entity__caption-wrapper')?.innerText || '';
                const description = item.querySelector('.pvs-list__outer-container')?.innerText || '';
                
                if (role || company) {
                    sections.experience.push({ role, company, duration, description });
                }
            });
        }

        // Extract education
        const educationSection = document.querySelector('#education');
        if (educationSection) {
            const educationItems = educationSection.querySelectorAll('.pvs-entity');
            educationItems.forEach(item => {
                const school = item.querySelector('.mr1 span[aria-hidden="true"]')?.innerText || '';
                const degree = item.querySelector('.t-14 span[aria-hidden="true"]')?.innerText || '';
                const duration = item.querySelector('.pvs-entity__caption-wrapper')?.innerText || '';
                
                if (school || degree) {
                    sections.education.push({ school, degree, duration });
                }
            });
        }

        // Extract skills
        const skillsSection = document.querySelector('#skills');
        if (skillsSection) {
            const skillItems = skillsSection.querySelectorAll('.pvs-entity');
            skillItems.forEach(item => {
                const skill = item.querySelector('.mr1 span[aria-hidden="true"]')?.innerText || '';
                if (skill) {
                    sections.skills.push(skill);
                }
            });
        }

        return this.formatResume(sections);
    },

    // Extract text from Naukri profile
    extractNaukri() {
        const sections = {
            name: document.querySelector('.name')?.innerText || '',
            headline: document.querySelector('.designation')?.innerText || '',
            experience: [],
            education: [],
            skills: []
        };

        // Extract experience
        const experienceItems = document.querySelectorAll('.experience-section .exp-item');
        experienceItems.forEach(item => {
            const role = item.querySelector('.designation')?.innerText || '';
            const company = item.querySelector('.org-name')?.innerText || '';
            const duration = item.querySelector('.duration')?.innerText || '';
            const description = item.querySelector('.description')?.innerText || '';
            
            sections.experience.push({ role, company, duration, description });
        });

        // Extract education
        const educationItems = document.querySelectorAll('.education-section .edu-item');
        educationItems.forEach(item => {
            const degree = item.querySelector('.degree')?.innerText || '';
            const school = item.querySelector('.institute')?.innerText || '';
            const year = item.querySelector('.year')?.innerText || '';
            
            sections.education.push({ school, degree, duration: year });
        });

        // Extract skills
        const skillItems = document.querySelectorAll('.skills-section .skill-item');
        skillItems.forEach(item => {
            sections.skills.push(item.innerText);
        });

        return this.formatResume(sections);
    },

    // Extract text from PDF viewer (Google Drive, browser PDF)
    extractPDF() {
        let text = '';

        // Try different PDF viewer selectors
        const selectors = [
            '.textLayer', // PDF.js
            '#viewer .page', // Google Drive PDF viewer
            '.pdfViewer .page', // Mozilla PDF viewer
            'embed[type="application/pdf"]' // Embedded PDFs
        ];

        for (const selector of selectors) {
            const elements = document.querySelectorAll(selector);
            if (elements.length > 0) {
                elements.forEach(el => {
                    text += el.innerText + '\n';
                });
                break;
            }
        }

        // If no PDF viewer found, try to get selected text
        if (!text) {
            text = window.getSelection().toString();
        }

        return text || 'Could not extract text from PDF. Please copy and paste the resume text.';
    },

    // Extract from Gmail attachment preview
    extractGmail() {
        let text = '';
        
        // Try to get text from attachment preview
        const previewElements = document.querySelectorAll('.aVy .a3s, .ii.gt .a3s');
        previewElements.forEach(el => {
            text += el.innerText + '\n';
        });

        if (!text) {
            // Try to get selected text
            text = window.getSelection().toString();
        }

        return text || 'Could not extract text from email. Please open the attachment or copy the text.';
    },

    // Format resume sections into text
    formatResume(sections) {
        let formatted = '';

        if (sections.name) {
            formatted += `Name: ${sections.name}\n`;
        }
        if (sections.headline) {
            formatted += `Current Role: ${sections.headline}\n\n`;
        }
        if (sections.about) {
            formatted += `About:\n${sections.about}\n\n`;
        }

        if (sections.experience && sections.experience.length > 0) {
            formatted += 'Experience:\n';
            sections.experience.forEach(exp => {
                formatted += `- ${exp.role} at ${exp.company} (${exp.duration})\n`;
                if (exp.description) {
                    formatted += `  ${exp.description}\n`;
                }
            });
            formatted += '\n';
        }

        if (sections.education && sections.education.length > 0) {
            formatted += 'Education:\n';
            sections.education.forEach(edu => {
                formatted += `- ${edu.degree} from ${edu.school} (${edu.duration})\n`;
            });
            formatted += '\n';
        }

        if (sections.skills && sections.skills.length > 0) {
            formatted += `Skills: ${sections.skills.join(', ')}\n`;
        }

        return formatted || 'No resume content found. Please ensure you are on a profile page or have a document open.';
    },

    // Main extraction function
    extract() {
        const url = window.location.href;
        
        if (url.includes('linkedin.com')) {
            return this.extractLinkedIn();
        } else if (url.includes('naukri.com')) {
            return this.extractNaukri();
        } else if (url.includes('drive.google.com') || url.includes('.pdf')) {
            return this.extractPDF();
        } else if (url.includes('mail.google.com')) {
            return this.extractGmail();
        } else {
            // Try generic text extraction
            const selectedText = window.getSelection().toString();
            if (selectedText) {
                return selectedText;
            }
            
            // Try to extract all text from body
            const bodyText = document.body.innerText;
            if (bodyText && bodyText.length > 100) {
                return bodyText.substring(0, 5000); // Limit to 5000 chars
            }
            
            return 'Could not extract resume from this page. Please select the text or use a supported platform (LinkedIn, Naukri, PDF).';
        }
    }
};

// Listen for messages from popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'extractResume') {
        try {
            const resumeText = ResumeExtractor.extract();
            sendResponse({ success: true, resumeText });
        } catch (error) {
            sendResponse({ success: false, error: error.message });
        }
    }
    return true; // Keep the message channel open for async response
});

// Add visual indicator when extension is active
const addIndicator = () => {
    if (!document.querySelector('.resumescan-indicator')) {
        const indicator = document.createElement('div');
        indicator.className = 'resumescan-indicator';
        indicator.innerHTML = '✓ ResumeScan AI Active';
        document.body.appendChild(indicator);
        
        setTimeout(() => {
            indicator.style.opacity = '0';
            setTimeout(() => indicator.remove(), 300);
        }, 3000);
    }
};

// Add indicator on supported pages
if (window.location.href.includes('linkedin.com') || 
    window.location.href.includes('naukri.com') ||
    window.location.href.includes('.pdf') ||
    window.location.href.includes('drive.google.com')) {
    addIndicator();
}