// Background service worker for API calls and subscription management

// Dynamic configuration
let BACKEND_URL = 'https://resumescan.ai'; // Default to production

// Backend configuration - EASY TO SWITCH!
const DEV_CONFIG = {
    // CHANGE THIS LINE TO SWITCH BACKENDS:
    // 'production' = Use resumescan.ai (recommended for testing)
    // 'local' = Use localhost:3006 (when running backend locally)  
    // 'auto' = Auto-detect (tries localhost, falls back to production)
    mode: 'production'  // ← CHANGE THIS TO SWITCH
};

// Try to fetch config from a known location
async function loadConfig() {
    try {
        // First, check if we're in production (Chrome Web Store)
        const manifest = chrome.runtime.getManifest();
        const isProduction = manifest.update_url?.includes('google.com');
        
        if (isProduction) {
            BACKEND_URL = 'https://resumescan.ai';
        } else {
            // Development mode - use config
            switch(DEV_CONFIG.mode) {
                case 'local':
                    BACKEND_URL = 'http://localhost:3006';
                    console.log('Using LOCAL backend');
                    break;
                    
                case 'auto':
                    // Auto-detect
                    try {
                        const controller = new AbortController();
                        const timeoutId = setTimeout(() => controller.abort(), 1000);
                        
                        const response = await fetch('http://localhost:3006/health', { 
                            method: 'GET',
                            mode: 'no-cors',
                            signal: controller.signal
                        });
                        clearTimeout(timeoutId);
                        BACKEND_URL = 'http://localhost:3006';
                        console.log('Local backend detected');
                    } catch (e) {
                        BACKEND_URL = 'https://resumescan.ai';
                        console.log('Using production backend (local not available)');
                    }
                    break;
                    
                case 'production':
                default:
                    BACKEND_URL = 'https://resumescan.ai';
                    console.log('Using PRODUCTION backend');
                    break;
            }
        }
        console.log('Backend URL:', BACKEND_URL);
    } catch (error) {
        console.log('Config load failed, using production:', 'https://resumescan.ai');
        BACKEND_URL = 'https://resumescan.ai';
    }
}

// Load config on startup
loadConfig();

const CONFIG = {
    FREE_DAILY_LIMIT: 5,
    get BACKEND_URL() {
        return BACKEND_URL;
    },
    API_ENDPOINT: '/api/score',
    AUTH_ENDPOINT: '/api/auth/login',
    SUBSCRIPTION_CHECK_URL: '/api/subscription/status',
    PADDLE_VENDOR_ID: '123456', // Replace with your Paddle vendor ID
    PADDLE_PRODUCT_ID: '789012' // Replace with your Paddle product ID
};

// Initialize storage
chrome.runtime.onInstalled.addListener(() => {
    chrome.storage.local.set({
        dailyScans: 0,
        lastResetDate: new Date().toDateString(),
        isPro: false,
        apiKey: '',
        userEmail: null,
        userId: null,
        deviceId: null
    });
    
    // Generate device ID if not exists
    generateDeviceId();
});

// Generate unique device fingerprint
async function generateDeviceId() {
    const data = await chrome.storage.local.get(['deviceId']);
    
    if (!data.deviceId) {
        // Create device fingerprint using available browser info
        // Note: screen object is not available in service workers
        const fingerprint = {
            // Platform info
            platform: navigator.platform || 'unknown',
            // Hardware concurrency
            cores: navigator.hardwareConcurrency || 0,
            // Memory (if available)
            memory: navigator.deviceMemory || 0,
            // User agent
            ua: navigator.userAgent,
            // Language
            lang: navigator.language,
            // Timezone
            tz: Intl.DateTimeFormat().resolvedOptions().timeZone,
            // Random component for uniqueness
            random: Math.random().toString(36).substring(2, 15),
            // Timestamp for uniqueness
            timestamp: Date.now()
        };
        
        // Create hash of fingerprint
        const fingerprintString = JSON.stringify(fingerprint);
        const encoder = new TextEncoder();
        const data = encoder.encode(fingerprintString);
        const hashBuffer = await crypto.subtle.digest('SHA-256', data);
        const hashArray = Array.from(new Uint8Array(hashBuffer));
        const deviceId = hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
        
        await chrome.storage.local.set({ deviceId });
        console.log('Device ID generated:', deviceId.substring(0, 16) + '...');
    }
    
    return data.deviceId;
}

// Google Authentication
async function authenticateWithGoogle() {
    try {
        // Get Google auth token
        const token = await chrome.identity.getAuthToken({ interactive: true });
        
        if (token) {
            // Get user info from Google
            const response = await fetch('https://www.googleapis.com/oauth2/v1/userinfo?alt=json', {
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            const userInfo = await response.json();
            
            // Store user info
            await chrome.storage.local.set({
                userEmail: userInfo.email,
                userId: userInfo.id,
                userName: userInfo.name,
                userPicture: userInfo.picture
            });
            
            // Register/login with backend
            const backendResponse = await fetch(`${CONFIG.BACKEND_URL}/api/auth/google`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    email: userInfo.email,
                    googleId: userInfo.id,
                    name: userInfo.name,
                    picture: userInfo.picture
                })
            });
            
            const result = await backendResponse.json();
            
            if (result.token) {
                await chrome.storage.local.set({
                    authToken: result.token,
                    isPro: result.isPro || false
                });
            }
            
            return { success: true, user: userInfo };
        }
    } catch (error) {
        console.error('Authentication failed:', error);
        return { success: false, error: error.message };
    }
}

// Get current user
async function getCurrentUser() {
    const data = await chrome.storage.local.get(['userEmail', 'userName', 'userPicture']);
    if (data.userEmail) {
        return {
            email: data.userEmail,
            name: data.userName,
            picture: data.userPicture
        };
    }
    return null;
}

// Logout
async function logout() {
    try {
        // Revoke the token
        const token = await chrome.identity.getAuthToken();
        if (token) {
            await chrome.identity.removeCachedAuthToken({ token });
            await fetch(`https://accounts.google.com/o/oauth2/revoke?token=${token}`);
        }
        
        // Clear storage
        await chrome.storage.local.set({
            userEmail: null,
            userId: null,
            userName: null,
            userPicture: null,
            authToken: null,
            isPro: false,
            dailyScans: 0
        });
        
        return { success: true };
    } catch (error) {
        console.error('Logout failed:', error);
        return { success: false, error: error.message };
    }
}

// Reset daily limit
async function checkAndResetDailyLimit() {
    const data = await chrome.storage.local.get(['dailyScans', 'lastResetDate']);
    const today = new Date().toDateString();
    
    if (data.lastResetDate !== today) {
        await chrome.storage.local.set({
            dailyScans: 0,
            lastResetDate: today
        });
        return 0;
    }
    
    return data.dailyScans || 0;
}

// Check if user can scan
async function canUserScan() {
    const data = await chrome.storage.local.get(['isPro', 'dailyScans']);
    
    if (data.isPro) {
        return { allowed: true, reason: 'pro' };
    }
    
    const currentScans = await checkAndResetDailyLimit();
    
    if (currentScans < CONFIG.FREE_DAILY_LIMIT) {
        return { allowed: true, reason: 'free', remaining: CONFIG.FREE_DAILY_LIMIT - currentScans };
    }
    
    return { allowed: false, reason: 'limit_reached' };
}

// Increment scan count
async function incrementScanCount() {
    const data = await chrome.storage.local.get(['dailyScans']);
    const newCount = (data.dailyScans || 0) + 1;
    await chrome.storage.local.set({ dailyScans: newCount });
    return newCount;
}

// Score resume against job description using GPT-4
async function scoreResume(resumeText, jobDescription, apiKey) {
    const prompt = `You are an expert recruiter and ATS system. Score this resume against the job description.

Job Description:
${jobDescription}

Resume:
${resumeText}

Provide a JSON response with:
1. overall_score: 0-100 match percentage
2. skills_score: 0-100 for technical skills match
3. experience_score: 0-100 for relevant experience
4. education_score: 0-100 for education requirements
5. insights: Array of 5 key insights (each with type: 'positive'/'negative'/'neutral' and text)
6. missing_keywords: Array of important keywords from JD missing in resume
7. recommendation: Brief recommendation (50 words max)

Return ONLY valid JSON, no other text.`;

    try {
        const response = await fetch(CONFIG.API_ENDPOINT, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${apiKey}`
            },
            body: JSON.stringify({
                model: 'gpt-4-turbo-preview',
                messages: [
                    { role: 'system', content: 'You are an expert ATS and recruiter. Always respond with valid JSON only.' },
                    { role: 'user', content: prompt }
                ],
                temperature: 0.3,
                max_tokens: 1000
            })
        });

        if (!response.ok) {
            throw new Error(`API error: ${response.status}`);
        }

        const data = await response.json();
        const content = data.choices[0].message.content;
        
        // Parse JSON response
        try {
            return JSON.parse(content);
        } catch (parseError) {
            // Try to extract JSON from the response
            const jsonMatch = content.match(/\{[\s\S]*\}/);
            if (jsonMatch) {
                return JSON.parse(jsonMatch[0]);
            }
            throw new Error('Invalid JSON response from API');
        }
    } catch (error) {
        console.error('Error scoring resume:', error);
        throw error;
    }
}

// Helper functions for mock scoring
function generateInsights(score, resumeText, jobDescription) {
    const insights = [];
    
    if (score >= 80) {
        insights.push({ type: 'positive', text: 'Excellent match with job requirements' });
        insights.push({ type: 'positive', text: 'Strong technical skills alignment' });
    } else if (score >= 60) {
        insights.push({ type: 'positive', text: 'Good overall match with core requirements' });
        insights.push({ type: 'neutral', text: 'Some skill gaps that can be addressed' });
    } else {
        insights.push({ type: 'neutral', text: 'Partial match with job requirements' });
        insights.push({ type: 'negative', text: 'Several key skills may need development' });
    }
    
    // Check for specific keywords
    if (jobDescription.toLowerCase().includes('aws') && !resumeText.toLowerCase().includes('aws')) {
        insights.push({ type: 'negative', text: 'AWS experience not clearly mentioned' });
    }
    if (jobDescription.toLowerCase().includes('leader') || jobDescription.toLowerCase().includes('lead')) {
        insights.push({ type: 'neutral', text: 'Highlight leadership experience more prominently' });
    }
    
    insights.push({ type: 'neutral', text: 'Consider adding quantifiable achievements' });
    
    return insights.slice(0, 5);
}

function findMissingKeywords(resumeText, jobDescription) {
    const importantKeywords = ['AWS', 'Terraform', 'CloudFormation', 'IAM', 'Docker', 'Kubernetes', 'Python', 'Java', 'Agile', 'Scrum', 'DevOps', 'CI/CD'];
    const resumeLower = resumeText.toLowerCase();
    const jdLower = jobDescription.toLowerCase();
    
    const missing = importantKeywords.filter(keyword => 
        jdLower.includes(keyword.toLowerCase()) && !resumeLower.includes(keyword.toLowerCase())
    );
    
    return missing.slice(0, 5);
}

function generateRecommendation(score) {
    if (score >= 85) {
        return 'Excellent candidate. Strongly recommend immediate phone screening to move forward quickly.';
    } else if (score >= 70) {
        return 'Strong candidate with good potential. Recommend phone screening to assess specific technical skills.';
    } else if (score >= 55) {
        return 'Promising candidate with some gaps. Consider for junior/mid-level positions or if willing to train.';
    } else {
        return 'May not be the best fit for this role. Consider for other openings or keep for future opportunities.';
    }
}

// Mock scoring function for testing (when no API key)
function mockScoreResume(resumeText, jobDescription) {
    // Simple keyword matching for demo
    const resumeWords = resumeText.toLowerCase().split(/\s+/);
    const jdWords = jobDescription.toLowerCase().split(/\s+/);
    
    // Key technical terms to look for
    const techKeywords = ['aws', 'cloud', 'terraform', 'migration', 'security', 'iac', 'devops', 'kubernetes', 'docker', 'python', 'java', 'agile'];
    const foundKeywords = techKeywords.filter(keyword => 
        resumeText.toLowerCase().includes(keyword) || jobDescription.toLowerCase().includes(keyword)
    );
    
    // Calculate base score from keyword matches
    const baseScore = 65 + (foundKeywords.length * 3);
    const matchPercentage = Math.min(95, Math.max(50, baseScore + Math.floor(Math.random() * 15)));
    
    // Ensure all scores are positive and reasonable
    const skillsScore = Math.min(100, Math.max(40, matchPercentage - 5 + Math.floor(Math.random() * 15)));
    const experienceScore = Math.min(100, Math.max(40, matchPercentage - 10 + Math.floor(Math.random() * 20)));
    const educationScore = Math.min(100, Math.max(60, 70 + Math.floor(Math.random() * 20)));
    
    return {
        overall_score: matchPercentage,
        skills_score: skillsScore,
        experience_score: experienceScore,
        education_score: educationScore,
        insights: generateInsights(matchPercentage, resumeText, jobDescription),
        missing_keywords: findMissingKeywords(resumeText, jobDescription),
        recommendation: generateRecommendation(matchPercentage)
    };
}

// Message handler
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'authenticateWithGoogle') {
        authenticateWithGoogle().then(sendResponse);
        return true;
    }
    
    if (request.action === 'getCurrentUser') {
        getCurrentUser().then(sendResponse);
        return true;
    }
    
    if (request.action === 'logout') {
        logout().then(sendResponse);
        return true;
    }
    
    if (request.action === 'checkCanScan') {
        canUserScan().then(sendResponse);
        return true;
    }
    
    if (request.action === 'scoreResume') {
        (async () => {
            try {
                // Check if user can scan
                const canScan = await canUserScan();
                if (!canScan.allowed) {
                    sendResponse({ success: false, error: 'Daily limit reached. Upgrade to Pro for unlimited scans.' });
                    return;
                }
                
                // Get user token and device ID
                const data = await chrome.storage.local.get(['authToken', 'deviceId']);
                const deviceId = data.deviceId || await generateDeviceId();
                
                let result;
                if (data.authToken) {
                    // Use secure backend API (YOUR server with YOUR API key)
                    try {
                        const response = await fetch(`${CONFIG.BACKEND_URL}${CONFIG.API_ENDPOINT}`, {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                                'Authorization': `Bearer ${data.authToken}`,
                                'X-Device-Id': deviceId
                            },
                            body: JSON.stringify({
                                resumeText: request.resumeText,
                                jobDescription: request.jobDescription,
                                deviceId: deviceId
                            })
                        });
                        
                        if (!response.ok) {
                            throw new Error('Backend API error');
                        }
                        
                        const responseData = await response.json();
                        result = responseData.result;
                    } catch (error) {
                        // Fallback to mock if backend fails
                        console.error('Backend error, using mock:', error);
                        result = mockScoreResume(request.resumeText, request.jobDescription);
                    }
                } else {
                    // Use mock scoring for demo (no login)
                    result = mockScoreResume(request.resumeText, request.jobDescription);
                }
                
                // Increment scan count
                await incrementScanCount();
                
                sendResponse({ success: true, result });
            } catch (error) {
                sendResponse({ success: false, error: error.message });
            }
        })();
        return true;
    }
    
    if (request.action === 'getUsageStats') {
        (async () => {
            const data = await chrome.storage.local.get(['dailyScans', 'isPro']);
            const currentScans = await checkAndResetDailyLimit();
            sendResponse({
                scansUsed: currentScans,
                scansLimit: CONFIG.FREE_DAILY_LIMIT,
                isPro: data.isPro || false
            });
        })();
        return true;
    }
    
    if (request.action === 'saveApiKey') {
        chrome.storage.local.set({ apiKey: request.apiKey });
        sendResponse({ success: true });
        return true;
    }
    
    if (request.action === 'upgradeToPro') {
        // Open Paddle checkout
        chrome.tabs.create({
            url: `https://checkout.paddle.com/checkout/product/${CONFIG.PADDLE_PRODUCT_ID}`
        });
        sendResponse({ success: true });
        return true;
    }
});

// Check subscription status periodically (for pro users)
async function checkSubscriptionStatus() {
    const data = await chrome.storage.local.get(['userEmail', 'isPro']);
    
    if (data.userEmail && data.isPro) {
        try {
            const response = await fetch(CONFIG.SUBSCRIPTION_CHECK_URL, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ email: data.userEmail })
            });
            
            if (response.ok) {
                const result = await response.json();
                await chrome.storage.local.set({ isPro: result.isActive });
            }
        } catch (error) {
            console.error('Error checking subscription:', error);
        }
    }
}

// Check subscription every hour
setInterval(checkSubscriptionStatus, 60 * 60 * 1000);

// Export functions for testing
if (typeof module !== 'undefined' && module.exports) {
    module.exports = { mockScoreResume, canUserScan, incrementScanCount };
}